/*
  STEP/DIR driver (HY-DIV-268N-5A apod.), Arduino UNO/Nano (ATmega328P)

  Zapojení:
    DIR+ -> D4
    PUL+ -> D5
    DIR-, PUL- -> GND

  Tlačítka:
    UP   -> D9 (INPUT_PULLUP)  (stisk = LOW)
    DOWN -> D8 (INPUT_PULLUP)  (stisk = LOW)
*/

#include <Arduino.h>

// Piny
const uint8_t STEP_PIN = 5;   // D5 = PD5
const uint8_t DIR_PIN  = 4;   // D4

const uint8_t UP_BTN   = 9;   // D9
const uint8_t DN_BTN   = 8;   // D8

// Nastavení pohybu (v krocích za sekundu)
// Tip: pro víc síly a méně ztracených kroků max třeba 6000–9000 místo 12500.
const float MAX_STEPS_PER_SEC = 7000.0f;     // max rychlost (steps/s)
const float ACCEL_STEPS_PER_S2 = 20000.0f;   // akcelerace (steps/s^2)

// Šířka pulsu STEP pro driver (µs)
const uint16_t STEP_PULSE_US = 10;

// Debounce tlačítek
const uint16_t DEBOUNCE_MS = 20;

// ===== interní proměnné pro timer =====
volatile bool     stepEnabled = false;
volatile uint16_t offTicks = 0;
volatile uint16_t onTicks  = 0;
volatile bool     stepHigh  = false;

// Rampa rychlosti
float currentSps = 0.0f;
float targetSps  = 0.0f;

uint32_t lastRampUs = 0;

// Pomocná: nastavit periodu timeru podle speed (steps/s)
void setStepRate(float stepsPerSec) {
  if (stepsPerSec <= 0.0f) {
    noInterrupts();
    stepEnabled = false;
    // stáhni STEP do LOW
    digitalWrite(STEP_PIN, LOW);
    stepHigh = false;
    interrupts();
    return;
  }

  // perioda jednoho kroku v mikrosekundách
  float periodUs = 1000000.0f / stepsPerSec;

  // zajisti, že se do periody vejde HIGH pulse
  if (periodUs < (float)STEP_PULSE_US + 2.0f) {
    periodUs = (float)STEP_PULSE_US + 2.0f;
  }

  float lowUs = periodUs - STEP_PULSE_US;

  // Timer1: prescaler 8 => 16MHz/8 = 2MHz => 0.5 µs na tick
  // ticks = us / 0.5 = us * 2
  uint16_t onT  = (uint16_t)(STEP_PULSE_US * 2);
  uint16_t offT = (uint16_t)(lowUs * 2);

  noInterrupts();
  onTicks  = (onT < 10) ? 10 : onT;    // pojistka
  offTicks = (offT < 10) ? 10 : offT;  // pojistka
  stepEnabled = true;
  interrupts();
}

// Timer1 Compare A ISR: střídá HIGH/LOW s různými intervaly (pulse + pauza)
ISR(TIMER1_COMPA_vect) {
  if (!stepEnabled) {
    return;
  }

  if (!stepHigh) {
    // jdeme do HIGH (pulse)
    // rychlé nastavení pro D5 (PD5) na AVR:
    PORTD |= _BV(PD5);
    stepHigh = true;
    OCR1A = onTicks;
  } else {
    // jdeme do LOW (zbytek periody)
    PORTD &= ~_BV(PD5);
    stepHigh = false;
    OCR1A = offTicks;
  }
  TCNT1 = 0;
}

bool readButtonStable(uint8_t pin) {
  // jednoduchý debounce: 2 čtení s odstupem
  if (digitalRead(pin) == LOW) {
    delay(DEBOUNCE_MS);
    return (digitalRead(pin) == LOW);
  }
  return false;
}

void setupTimer1() {
  // STEP pin jako výstup a LOW
  pinMode(STEP_PIN, OUTPUT);
  digitalWrite(STEP_PIN, LOW);

  // Timer1 CTC, prescaler 8
  noInterrupts();
  TCCR1A = 0;
  TCCR1B = 0;
  TCCR1B |= _BV(WGM12);   // CTC
  TCCR1B |= _BV(CS11);    // prescaler 8
  TCNT1 = 0;
  OCR1A = 2000;           // start hodnota (neběží, dokud stepEnabled)
  TIMSK1 |= _BV(OCIE1A);  // enable compare A interrupt
  interrupts();
}

void setup() {
  pinMode(DIR_PIN, OUTPUT);
  pinMode(UP_BTN, INPUT_PULLUP);
  pinMode(DN_BTN, INPUT_PULLUP);

  setupTimer1();

  lastRampUs = micros();
}

void loop() {
  // Určení směru a cílové rychlosti podle tlačítek
  bool up = (digitalRead(UP_BTN) == LOW);
  bool dn = (digitalRead(DN_BTN) == LOW);

  if (up && !dn) {
    digitalWrite(DIR_PIN, LOW);
    targetSps = MAX_STEPS_PER_SEC;
  } else if (dn && !up) {
    digitalWrite(DIR_PIN, HIGH);
    targetSps = MAX_STEPS_PER_SEC;
  } else {
    targetSps = 0.0f;
  }

  // Rampa rychlosti (akcelerace/decelerace)
  uint32_t nowUs = micros();
  float dt = (nowUs - lastRampUs) / 1000000.0f;
  if (dt < 0.001f) dt = 0.001f; // pojistka
  lastRampUs = nowUs;

  float maxDelta = ACCEL_STEPS_PER_S2 * dt;

  if (currentSps < targetSps) {
    currentSps += maxDelta;
    if (currentSps > targetSps) currentSps = targetSps;
  } else if (currentSps > targetSps) {
    currentSps -= maxDelta;
    if (currentSps < targetSps) currentSps = targetSps;
  }

  setStepRate(currentSps);

  // malé odlehčení CPU
  delay(1);
}
